/*****************************************************************************
* | File      	:   EPD_4in2b_V2.c
* | Author      :   Waveshare team
* | Function    :   4.2inch e-paper b V2
* | Info        :
*----------------
* |	This version:   V1.0
* | Date        :   2020-11-27
* | Info        :
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documnetation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to  whom the Software is
# furished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS OR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
#
******************************************************************************/
#include "EPD_4in2b_V2.h"
#include "Debug.h"

/******************************************************************************
function :	Software reset
parameter:
******************************************************************************/
static void EPD_4IN2B_V2_Reset(void)
{
	DEV_Digital_Write(EPD_RST_PIN, 1);
	DEV_Delay_ms(200);
	DEV_Digital_Write(EPD_RST_PIN, 0);
	DEV_Delay_ms(2);
	DEV_Digital_Write(EPD_RST_PIN, 1);
	DEV_Delay_ms(200);
}

/******************************************************************************
function :	send command
parameter:
     Reg : Command register
******************************************************************************/
static void EPD_4IN2B_V2_SendCommand(UBYTE Reg)
{
	DEV_Digital_Write(EPD_DC_PIN, 0);
	DEV_Digital_Write(EPD_CS_PIN, 0);
	DEV_SPI_WriteByte(Reg);
	DEV_Digital_Write(EPD_CS_PIN, 1);
}

/******************************************************************************
function :	send data
parameter:
    Data : Write data
******************************************************************************/
static void EPD_4IN2B_V2_SendData(UBYTE Data)
{
	DEV_Digital_Write(EPD_DC_PIN, 1);
	DEV_Digital_Write(EPD_CS_PIN, 0);
	DEV_SPI_WriteByte(Data);
	DEV_Digital_Write(EPD_CS_PIN, 1);
}

/******************************************************************************
function :	Wait until the busy_pin goes LOW
parameter:
******************************************************************************/
void EPD_4IN2B_V2_ReadBusy(void)
{
    Debug("e-Paper busy\r\n");
    do{
        EPD_4IN2B_V2_SendCommand(0x71);
		DEV_Delay_ms(50);
    }while(!(DEV_Digital_Read(EPD_BUSY_PIN)));
    Debug("e-Paper busy release\r\n");
    DEV_Delay_ms(50);
}

/******************************************************************************
function :	Turn On Display
parameter:
******************************************************************************/
void EPD_4IN2B_V2_TurnOnDisplay(void)
{
	EPD_4IN2B_V2_SendCommand(0x12); // DISPLAY_REFRESH
	DEV_Delay_ms(100);
	EPD_4IN2B_V2_ReadBusy();
}

/******************************************************************************
function :	Initialize the e-Paper register
parameter:
******************************************************************************/
void EPD_4IN2B_V2_Init(void)
{
    EPD_4IN2B_V2_Reset();
    
    EPD_4IN2B_V2_SendCommand(0x04); 
    EPD_4IN2B_V2_ReadBusy();

    EPD_4IN2B_V2_SendCommand(0x00);
    EPD_4IN2B_V2_SendData(0x0f);
}

/******************************************************************************
function :	Clear screen
parameter:
******************************************************************************/
void EPD_4IN2B_V2_Clear(void)
{
	UWORD Width, Height;
	Width = (EPD_4IN2B_V2_WIDTH % 8 == 0)? (EPD_4IN2B_V2_WIDTH / 8 ): (EPD_4IN2B_V2_WIDTH / 8 + 1);
	Height = EPD_4IN2B_V2_HEIGHT;

	EPD_4IN2B_V2_SendCommand(0x10);
	for (UWORD j = 0; j < Height; j++) {
		for (UWORD i = 0; i < Width; i++) {
			EPD_4IN2B_V2_SendData(0xFF);
		}
	}

	EPD_4IN2B_V2_SendCommand(0x13);
	for (UWORD j = 0; j < Height; j++) {
		for (UWORD i = 0; i < Width; i++) {
			EPD_4IN2B_V2_SendData(0xFF);
		}
	}

	EPD_4IN2B_V2_TurnOnDisplay();
}

/******************************************************************************
function :	Sends the image buffer in RAM to e-Paper and displays
parameter:
******************************************************************************/
void EPD_4IN2B_V2_Display(const UBYTE *blackimage, const UBYTE *ryimage)
{
	UWORD Width, Height;
	Width = (EPD_4IN2B_V2_WIDTH % 8 == 0)? (EPD_4IN2B_V2_WIDTH / 8 ): (EPD_4IN2B_V2_WIDTH / 8 + 1);
	Height = EPD_4IN2B_V2_HEIGHT;

	EPD_4IN2B_V2_SendCommand(0x10);
	for (UWORD j = 0; j < Height; j++) {
		for (UWORD i = 0; i < Width; i++) {
			EPD_4IN2B_V2_SendData(blackimage[i + j * Width]);
		}
	}

	EPD_4IN2B_V2_SendCommand(0x13);
	for (UWORD j = 0; j < Height; j++) {
		for (UWORD i = 0; i < Width; i++) {
			EPD_4IN2B_V2_SendData(ryimage[i + j * Width]);
		}
	}

	EPD_4IN2B_V2_TurnOnDisplay();
}

/******************************************************************************
function :	Sends the image buffer Half image
parameter:
******************************************************************************/
void EPD_4IN2B_V2_SendHalfBimage(char TopOrBottom, const UBYTE *image)
{
	UWORD Width, Height;
	Width = (EPD_4IN2B_V2_WIDTH % 8 == 0)? (EPD_4IN2B_V2_WIDTH / 8 ): (EPD_4IN2B_V2_WIDTH / 8 + 1);
	Height = EPD_4IN2B_V2_HEIGHT;

	if(TopOrBottom == 0) { //the top half screen
		EPD_4IN2B_V2_SendCommand(0x10);
		Debug("send the top half black screen\r\n");
		for (UWORD j = 0; j < Height / 2; j++) {
			for (UWORD i = 0; i < Width; i++) {
				EPD_4IN2B_V2_SendData(image[i + j * Width]);
			}
		}
	} else if(TopOrBottom == 1) { //the Bottom half screen
		Debug("send the the Bottom half black screen\r\n");
		for (UWORD j = 0; j < Height / 2; j++) {
			for (UWORD i = 0; i < Width; i++) {
				EPD_4IN2B_V2_SendData(image[i + j * Width]);
			}
		}
	}
}

void EPD_4IN2B_V2_SendHalfRYimage(char TopOrBottom, const UBYTE *image)
{
	UWORD Width, Height;
	Width = (EPD_4IN2B_V2_WIDTH % 8 == 0)? (EPD_4IN2B_V2_WIDTH / 8 ): (EPD_4IN2B_V2_WIDTH / 8 + 1);
	Height = EPD_4IN2B_V2_HEIGHT;

	if(TopOrBottom == 0) { //the top half screen
		EPD_4IN2B_V2_SendCommand(0x13);
		Debug("send the top half red screen\r\n");
		for (UWORD j = 0; j < Height / 2; j++) {
			for (UWORD i = 0; i < Width; i++) {
				EPD_4IN2B_V2_SendData(image[i + j * Width]);
			}
		}
	} else if(TopOrBottom == 1) { //the Bottom half screen
		Debug("send the top half red screen\r\n");
		for (UWORD j = 0; j < Height / 2; j++) {
			for (UWORD i = 0; i < Width; i++) {
				EPD_4IN2B_V2_SendData(image[i + j * Width]);
			}
		}
	}
}

/******************************************************************************
function :	Enter sleep mode
parameter:
******************************************************************************/
void EPD_4IN2B_V2_Sleep(void)
{
    EPD_4IN2B_V2_SendCommand(0X50);
    EPD_4IN2B_V2_SendData(0xf7);		//border floating	

    EPD_4IN2B_V2_SendCommand(0X02);  	//power off
    EPD_4IN2B_V2_ReadBusy(); //waiting for the electronic paper IC to release the idle signal
    EPD_4IN2B_V2_SendCommand(0X07);  	//deep sleep
    EPD_4IN2B_V2_SendData(0xA5);
}
